import { Player, system, world } from "@minecraft/server";
import { DiamondForms } from "../forms/form";
import { ActionFormData, MessageFormData, ModalFormData } from "@minecraft/server-ui";
import { changeGM, giveTag, playSound, rawText, removeTag, runCommand, runCommands, tdpGive, teleportAction, tellRaw } from "./tdpTools";
import { ModalForms } from "../forms/modal";
import { applyScoreModal, changeGMModal, giveEffectModal, giveTagModal, removeTagModal, sendMessageModal, summonMobToPlayerModal, tdpModalGive, tpPlayerOrCoordsModal } from "./tdpModalTools";
import { injectActionValues } from "../utils/injectValue";
import { DEBUG } from "../config";

// A lot of this was inspired by AxisCube's Form Builder, but modified to be more advanced and up-to-date

/**
 * The different events that can be triggered
 * @important AxisCube made this actionChain, I take no credit
 * @param {Events[]} chain - A chain of events that will be executed
 * @param {Player} player - The player that triggered the events
 */
export function actionChain(chain, player) {
    let name = player.name;
    for (let i in chain) {
        let toValue = chain[i].value;
        switch (chain[i].type) {
            case 'eval':
                eval(toValue);
                break;
            case 'form':
                formBuilder(toValue, player);
                break;
            case 'sound':
                playSound(toValue, player);
                break;
            case 'give':
                tdpGive(toValue, player);
                break;
            case 'cmds':
                runCommands(toValue, player);
                break;
            case 'say':
                rawText(toValue, name);
                break;
            case 'tellraw':
                tellRaw(toValue, name);
                break;
            case 'cmd':
                runCommand(toValue, player);
                break;
            case 'add_tag':
                giveTag(toValue, player);
                break;
            case 'remove_tag':
                removeTag(toValue, player);
                break;
            case 'gamemode':
                changeGM(toValue, player, valueMap);
                break;
            case 'teleport':
            case 'tp':
                teleportAction(toValue, player);
                break;
        }
    }
}

/**
 * The different events that can be triggered
 * @param {Events[]} chain - A chain of events that will be executed
 * @param {Player} player - The player that triggered the events
 */
export function actionChainModal(chain, player, valueMap = {}, action) {
    let name = player.name;
    for (let i in chain) {
        let toValue = chain[i].value;
        switch (chain[i].type) {
            case 'form':
                formBuilder(toValue, player);
                break;
            case 'give':
                tdpModalGive(toValue, player);
                break;
            case 'add_tag':
                giveTagModal(toValue, player);
                break;
            case 'remove_tag':
                removeTagModal(toValue, player);
                break;
            case 'effect':
                giveEffectModal(toValue, player);
                break;
            case 'scoreboard':
                applyScoreModal(toValue, player);
                break;
            case 'message':
                sendMessageModal(toValue, player);
                break;
            case 'confirm':
                import("./confirmationFormBuilder").then(module => {
                    module.confirmationForm(player, action[0], valueMap);
                });
                break;
            case 'gamemode':
                changeGMModal(toValue, player);
            case 'teleport':
            case 'tp':
                tpPlayerOrCoordsModal(toValue, player);
                break;
            case 'summon':
                summonMobToPlayerModal(toValue, player);
                break;
        }
    }
}


/**
 * The main function that builds all the forms
 * @param {string} systemName - This is the name of the forms
 * @param {Player} player - The player that is opening the forms
 */
export function formBuilder(systemName, player) {
    system.run(() => {
        let formData = (DiamondForms[systemName]||ModalForms[systemName]);

        if (formData.type == undefined || formData.type == 'action') {
            if (formData.onOpen != undefined) {
                actionChain(formData.onOpen, player);
            }

            const actionForm = new ActionFormData();
            actionForm.title(formData.title);
            actionForm.body(formData?.body ?? "");
            let elements = formData.elements;
            let elementIndex = -1;
            for (let i in elements) {
                let thisElem = elements[i];
                switch (thisElem.type) {
                    case "divider":
                        actionForm.divider();
                        break;
                    case "header":
                        actionForm.header(thisElem.headerText);
                        break;
                    case "label":
                        actionForm.label(thisElem.labelText);
                        break;
                    case "button":
                        elementIndex++;
                        actionForm.button(thisElem.buttonName, thisElem?.icon);
                        thisElem._index = elementIndex;
                        break;
                }        
            }
            actionForm.show(player).then(r => {
                const buttons = elements.filter(e => e.type === "button");
                const selected = buttons[r.selection];
                if (r.selection != undefined) {
                    actionChain(selected.onClick, player);
                } else if (formData.onClose != undefined) {
                    actionChain(formData.onClose, player);
                }
            })
        } else if (formData.type == "modal") {
            if (formData.onOpen != undefined) {
                actionChain(formData.onOpen, player);
            }

            const modalForm = new ModalFormData()
            modalForm.title(formData.title);
            
            let players = world.getPlayers();
            const playernames = Array.from(players, plr => plr.name);
            players = [...players]
            let elements = formData.elements;
            for (let i in elements) {
                const thisElem = elements[i];
                switch (thisElem.type) {
                    case "divider":
                        modalForm.divider();
                        break;
                    case "header":
                        modalForm.header(thisElem.label);
                        break;
                    case "label":
                        modalForm.label(thisElem.label);
                        break;
                    case "dropdown":
                        modalForm.dropdown(thisElem.label, thisElem.items, {
                            defaultValueIndex: thisElem?.default,
                            tooltip: thisElem?.tooltip
                        });
                        break;
                    case "player": {
                        modalForm.dropdown(thisElem.label, playernames);
                        break;
                    }
                    case "slider":
                        let range = thisElem.range;
                        modalForm.slider(thisElem.label, range.min, range.max, {
                            valueStep: range?.step ?? 1,
                            defaultValue: range?.default
                        });
                        break;
                    case "toggle":
                        modalForm.toggle(thisElem.label, {
                            defaultValue: thisElem?.default
                        });
                        break;
                    case "textField":
                        modalForm.textField(thisElem.label, thisElem.placeholder, {
                            defaultValue: thisElem?.default
                        });
                        break;
                    case "submitButton":
                        modalForm.submitButton(thisElem.label);
                        break;
                }
            };
            modalForm.show(player).then(r => {
                let formValue = r.formValues;
                if (r.canceled) {
                    actionChain(formData.onClose, player);
                    return;
                }
                const valueMap = {};
                for (let i in elements) {
                    const thisElem = elements[i];
                    const key = thisElem.value;
                    const result = formValue[i];

                    if (!key || thisElem.type === "submitButton") continue;

                    switch (thisElem.type) {
                        case "player":
                            valueMap[key] = players[result]?.name ?? "unknown";
                            break;
                        case "dropdown":
                            valueMap[key] = thisElem.items?.[result] ?? result;
                            break;
                        default:
                            valueMap[key] = result;
                            break;
                    }
                }

                const submitElem = elements.find(e => e.type === "submitButton");
                if (!submitElem?.onSubmit) return;

                const injectedActions = submitElem.onSubmit.map(action => injectActionValues(action, valueMap));

                if (DEBUG) console.log('The Injected Action: \n',JSON.stringify(injectedActions));
                if (DEBUG) console.log('The Value Map: \n', JSON.stringify(valueMap));
                actionChainModal(injectedActions, player, valueMap, injectedActions);
            })
        } else if (formData.type == "message") {
            if (formData.onOpen != undefined) {
                actionChain(formData.onOpen, player);
            }

            const messageForm = new MessageFormData();
            messageForm.title(formData.title);
            messageForm.body(formData?.body ?? "");
            let elements = formData.elements;
            for (let i in elements) {
                let thisElem = elements[i];
                switch (thisElem.type) {
                    case "button1":
                        messageForm.button1(thisElem?.label ?? "No");
                        break;
                    case "button2":
                        messageForm.button2(thisElem?.label ?? "Yes");
                        break;
                }
            }
            messageForm.show(player).then(r => {
                if (r.selection != undefined) {
                    actionChain(elements[r.selection].onClick, player);
                } else if (formData.onClose != undefined) {
                    actionChain(formData.onClose, player);
                }
            })
        }
    })
}